/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bcel.verifier.structurals;

import java.util.ArrayList;

import org.apache.bcel.generic.InstructionHandle;

/**
 * An InstructionContext offers convenient access to information like control flow successors and such.
 */
public interface InstructionContext {

    /**
     * This method symbolically executes the Instruction held in the InstructionContext. It "merges in" the incoming
     * execution frame situation (see The Java Virtual Machine Specification, 2nd edition, page 146). By so doing, the
     * outgoing execution frame situation is calculated.
     *
     * This method is JustIce-specific and is usually of no sense for users of the ControlFlowGraph class. They should use
     * getInstruction().accept(Visitor), possibly in conjunction with the ExecutionVisitor.
     *
     * @param inFrame the incoming frame.
     * @param executionPredecessors the execution predecessors.
     * @param icv the instruction constraint visitor.
     * @param ev the execution visitor.
     * @see ControlFlowGraph
     * @see ExecutionVisitor
     * @see #getOutFrame(ArrayList)
     * @return true - if and only if the "outgoing" frame situation changed from the one before execute()ing.
     */
    boolean execute(Frame inFrame, ArrayList<InstructionContext> executionPredecessors, InstConstraintVisitor icv, ExecutionVisitor ev);

    /**
     * Returns the exception handlers that protect this instruction. They are special control flow successors.
     *
     * @return the exception handlers.
     */
    ExceptionHandler[] getExceptionHandlers();

    /**
     * Gets the incoming frame.
     *
     * @return the incoming frame.
     */
    Frame getInFrame();

    /**
     * Returns the InstructionHandle this InstructionContext is wrapped around.
     *
     * @return The InstructionHandle this InstructionContext is wrapped around.
     */
    InstructionHandle getInstruction();

    /**
     * This method returns the outgoing execution frame situation; therefore <B>it has to be calculated by execute(Frame,
     * ArrayList) first.</B>
     *
     * @param executionPredecessors the execution predecessors.
     * @return the outgoing frame.
     * @see #execute(Frame, ArrayList, InstConstraintVisitor, ExecutionVisitor)
     */
    Frame getOutFrame(ArrayList<InstructionContext> executionPredecessors);

    /**
     * Returns the usual control flow successors.
     *
     * @return the successors.
     * @see #getExceptionHandlers()
     */
    InstructionContext[] getSuccessors();

    /**
     * The getTag and setTag methods may be used for temporary flagging, such as graph coloring. Nothing in the
     * InstructionContext object depends on the value of the tag. JustIce does not use it.
     *
     * @return the tag value.
     * @see #setTag(int tag)
     */
    int getTag();

    /**
     * The getTag and setTag methods may be used for temporary flagging, such as graph coloring. Nothing in the
     * InstructionContext object depends on the value of the tag. JustIce does not use it.
     *
     * @param tag the tag value.
     * @see #getTag()
     */
    void setTag(int tag);
}
